"use client";

import {
  Beer,
  IceCream,
  Pizza,
  Sandwich,
  MessageCircle,
  type LucideIcon,
} from 'lucide-react';
import Image from 'next/image';
import Autoplay from "embla-carousel-autoplay";

import { categories } from '@/lib/menu-data';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { ProductCard } from '@/components/product-card';
import { MenuHeader } from '@/components/menu-header';
import { CartSheet } from '@/components/cart-sheet';
import { useRestaurantInfo } from '@/hooks/use-restaurant-info';
import Link from 'next/link';
import { Card, CardContent } from "@/components/ui/card"
import {
  Carousel,
  CarouselContent,
  CarouselItem,
  CarouselNext,
  CarouselPrevious,
} from "@/components/ui/carousel"

const iconMap: { [key: string]: LucideIcon } = {
  Pizzas: Pizza,
  Bebidas: Beer,
  Lanches: Sandwich,
  Sobremesas: IceCream,
};

export default function Home() {
  const { restaurantInfo } = useRestaurantInfo();
  const { carouselImages } = restaurantInfo;

  return (
    <div className="flex flex-col min-h-screen bg-background font-body">
      <MenuHeader />
      <main className="flex-grow p-4 md:p-6 max-w-7xl mx-auto w-full">
        <h1 className="text-3xl md:text-4xl font-bold font-headline text-center mb-2">
          Nosso Cardápio
        </h1>
        <p className="text-center text-muted-foreground mb-6 md:mb-8">
          Escolha uma categoria para começar a montar seu pedido.
        </p>

        <div className="mb-8">
          <Carousel 
            className="w-full"
            plugins={[
              Autoplay({
                delay: 4000,
              }),
            ]}
            opts={{
              loop: true,
            }}
          >
            <CarouselContent>
              {carouselImages.map((image) => (
                <CarouselItem key={image.id}>
                  <Card>
                    <CardContent className="relative flex aspect-[2.4/1] items-center justify-center p-0 overflow-hidden rounded-lg">
                       <Image 
                          src={image.imageUrl} 
                          alt={image.description} 
                          fill
                          className="object-cover"
                          data-ai-hint={image.imageHint}
                        />
                    </CardContent>
                  </Card>
                </CarouselItem>
              ))}
            </CarouselContent>
            <CarouselPrevious className="absolute left-2 top-1/2 -translate-y-1/2 z-10 hidden sm:flex" />
            <CarouselNext className="absolute right-2 top-1/2 -translate-y-1/2 z-10 hidden sm:flex" />
          </Carousel>
        </div>


        {categories.length > 0 ? (
          <Tabs defaultValue={categories[0].id} className="w-full">
            <TabsList className="grid w-full grid-cols-2 md:grid-cols-4 h-auto">
              {categories.map((category) => {
                const Icon = iconMap[category.name];
                return (
                  <TabsTrigger
                    key={category.id}
                    value={category.id}
                    className="flex flex-col sm:flex-row gap-2 h-12 data-[state=inactive]:hover:bg-accent/50 data-[state=inactive]:hover:text-accent-foreground"
                  >
                    {Icon && <Icon className="h-5 w-5" />}
                    {category.name}
                  </TabsTrigger>
                );
              })}
            </TabsList>
            {categories.map((category) => (
              <TabsContent key={category.id} value={category.id}>
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4 md:gap-6 mt-6">
                  {category.products.map((product) => (
                    <ProductCard key={product.id} product={product} />
                  ))}
                </div>
              </TabsContent>
            ))}
          </Tabs>
        ) : (
          <p className="text-center text-muted-foreground mt-12">
            Nenhum item no cardápio no momento.
          </p>
        )}
      </main>
      <footer className="w-full mt-auto bg-card border-t py-6">
        <div className="max-w-7xl mx-auto flex flex-col md:flex-row justify-center items-center gap-4 px-4 text-center">
          <p className="text-sm text-muted-foreground">
            &copy; {new Date().getFullYear()} {restaurantInfo.name}. {restaurantInfo.footerText}
          </p>
          <div className="flex gap-4 items-center">
            <Link href="/terms" className="text-sm text-muted-foreground hover:text-primary transition-colors">
              Termos de Uso
            </Link>
            <Link href="/privacy" className="text-sm text-muted-foreground hover:text-primary transition-colors">
              Política de Privacidade
            </Link>
             <Link 
              href={`https://wa.me/${restaurantInfo.whatsappNumber}`} 
              target="_blank" 
              rel="noopener noreferrer"
              className="text-sm text-muted-foreground hover:text-primary transition-colors flex items-center gap-1"
            >
              <MessageCircle className="h-4 w-4" />
              WhatsApp
            </Link>
          </div>
        </div>
      </footer>
      <CartSheet />
    </div>
  );
}
